let current = 0;
let editor = null;
let score = 0;
let isRunning = false;
let cheatCounter = 0;
let currentPage = 0;
const questionsPerPage = 5;

/**
 * Menampilkan pesan peringatan di konsol.
 * @param {string} message
 */
function logWarning(message) {
    const consoleEl = document.getElementById('console');
    const warningHTML = `<div class="console-line console-warning"><i class="fa-solid fa-triangle-exclamation mr-2"></i>${message}</div>`;
    consoleEl.innerHTML = warningHTML + consoleEl.innerHTML;
}

function initEditor() {
    editor = ace.edit('editor');
    editor.setTheme('ace/theme/dracula');
    editor.session.setMode('ace/mode/javascript');
    editor.setOptions({
        fontSize: '15px',
        showPrintMargin: false,
        showLineNumbers: true,
        showGutter: true,
        highlightActiveLine: true,
        enableBasicAutocompletion: true,
        enableLiveAutocompletion: true
    });

    const editorEl = document.getElementById('editor');
    editorEl.addEventListener('paste', (e) => {
        e.preventDefault();
        logWarning('Aktivitas menempel (paste) tidak diizinkan. Silakan ketik solusi Anda secara manual.');
    });
}

function setModeForLang(lang) {
    if (lang === 'js') editor.session.setMode('ace/mode/javascript');
    else if (lang === 'py') editor.session.setMode('ace/mode/python');
    else if (lang === 'cpp') editor.session.setMode('ace/mode/c_cpp');
    else if (lang === 'csharp') editor.session.setMode('ace/mode/csharp');
    else if (lang === 'rb') editor.session.setMode('ace/mode/ruby');
    else if (lang === 'go') editor.session.setMode('ace/mode/golang');
}

function renderQuestion(i) {
    const q = QUESTIONS[i];
    document.getElementById('q-id').textContent = q.id;
    document.getElementById('q-title').textContent = q.title;
    document.getElementById('q-desc').textContent = q.description;
    document.getElementById('q-expected').textContent = q.expected_output;

    const lang = document.getElementById('lang').value;
    let tpl = q.template_js;
    if (lang === 'py') tpl = q.template_py;
    else if (lang === 'cpp') tpl = q.template_cpp;
    else if (lang === 'csharp') tpl = q.template_csharp;
    else if (lang === 'rb') tpl = q.template_rb;
    else if (lang === 'go') tpl = q.template_go;

    editor.setValue(tpl, -1);
    setModeForLang(lang);
    highlightList();
}

function highlightList() {
    const ul = document.getElementById('question-list');
    ul.innerHTML = '';
    
    const totalPages = Math.ceil(QUESTIONS.length / questionsPerPage);
    const startIndex = currentPage * questionsPerPage;
    const endIndex = startIndex + questionsPerPage;
    const paginatedQuestions = QUESTIONS.slice(startIndex, endIndex);

    paginatedQuestions.forEach((q, index) => {
        const originalIndex = startIndex + index;
        const li = document.createElement('li');
        li.className = `p-3 rounded-lg transition-all cursor-pointer flex items-center gap-3 border border-transparent ${
            originalIndex === current ? 'bg-white/10 border-[color:var(--accent)]' : 'hover:bg-white/5'
        }`;

        li.innerHTML = `
            <div class="w-8 h-8 rounded-md flex items-center justify-center text-white ${originalIndex === current ? 'bg-[color:var(--accent)]' : 'bg-gray-700'}">
                <span class="text-sm font-medium">${q.id}</span>
            </div>
            <span class="truncate">${q.title}</span>
        `;

        li.onclick = () => {
            current = originalIndex;
            renderQuestion(originalIndex);
        };
        ul.appendChild(li);
    });

    const pageInfo = document.getElementById('page-info');
    const prevPageBtn = document.getElementById('page-prev-btn');
    const nextPageBtn = document.getElementById('page-next-btn');

    pageInfo.textContent = `Page ${currentPage + 1} of ${totalPages}`;
    prevPageBtn.disabled = currentPage === 0;
    nextPageBtn.disabled = currentPage >= totalPages - 1;
}

async function apiRun(lang, code) {
    const res = await fetch(location.pathname + '?action=run', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ lang, code })
    });
    return await res.json();
}

async function runHandler() {
    if (isRunning) return;
    isRunning = true;

    const code = editor.getValue();
    const lang = document.getElementById('lang').value;

    const runBtn = document.getElementById('run-btn');
    runBtn.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Running...';
    runBtn.disabled = true;

    document.getElementById('console').innerHTML = `<div class="console-line pulse"><i class="fa-solid fa-gears mr-2"></i>Executing code...</div>`;

    try {
        const j = await apiRun(lang, code);
        const r = j.result;
        let consoleContent = '';

        if (!j.ok) {
            consoleContent = `<div class="console-line console-error"><i class="fa-solid fa-bug mr-2"></i>Error: ${j.msg}</div>`;
        } else {
            if (r.timed) {
                consoleContent += `<div class="console-line console-error"><i class="fa-solid fa-clock mr-2"></i>TIMEOUT - Proses eksekusi terlalu lama</div>`;
            }
            if (r.stdout) {
                const lines = r.stdout.split('\n');
                lines.forEach(line => {
                    if (line.trim() !== '') {
                        consoleContent += `<div class="console-line">${line}</div>`;
                    }
                });
            }
            if (r.stderr) {
                const errorLines = r.stderr.split('\n');
                errorLines.forEach(line => {
                    if (line.trim() !== '') {
                        consoleContent += `<div class="console-line console-error"><i class="fa-solid fa-circle-exclamation mr-2"></i>${line}</div>`;
                    }
                });
            }
            if (!r.stdout && !r.stderr) {
                consoleContent = `<div class="console-line"><i class="fa-solid fa-info-circle mr-2"></i>Tidak ada output yang dihasilkan</div>`;
            }
        }
        document.getElementById('console').innerHTML = consoleContent;

    } catch (error) {
        document.getElementById('console').innerHTML = `<div class="console-line console-error"><i class="fa-solid fa-circle-exclamation mr-2"></i>Network error: ${error.message}</div>`;
    } finally {
        runBtn.innerHTML = '<i class="fa-solid fa-play"></i> Run Code';
        runBtn.disabled = false;
        isRunning = false;
    }
}

async function submitHandler() {
    if (isRunning) return;
    isRunning = true;

    const code = editor.getValue();
    const lang = document.getElementById('lang').value;

    const submitBtn = document.getElementById('submit-btn');
    submitBtn.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Checking...';
    submitBtn.disabled = true;

    document.getElementById('console').innerHTML = `<div class="console-line pulse"><i class="fa-solid fa-circle-check mr-2"></i>Checking your solution...</div>`;

    try {
        const res = await fetch(location.pathname + '?action=run', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                lang,
                code,
                questionId: QUESTIONS[current].id,
                submit: true
            })
        });
        if (!res.ok) {
            throw new Error(`Server responded with status ${res.status}`);
        }
        const j = await res.json();
        
        let consoleContent = '';

        if (j.correct) {
            confetti({
                particleCount: 100,
                spread: 70,
                origin: { y: 0.6 }
            });
            consoleContent += `<div class="console-line console-success"><i class="fa-solid fa-check-circle mr-2"></i>Correct! +${j.points} points</div>`;
            consoleContent += `<div class="console-line"><i class="fa-solid fa-arrow-right mr-2"></i>Output: ${j.stdout}</div>`;
            document.getElementById('question-card').classList.add('glow-success');
        } else {
            consoleContent += `<div class="console-line console-error shake"><i class="fa-solid fa-face-angry mr-2"></i>Wrong answer!</div>`;
            consoleContent += `<div class="console-line"><i class="fa-solid fa-lightbulb mr-2"></i>Expected: ${QUESTIONS[current].expected_output}</div>`;
            consoleContent += `<div class="console-line"><i class="fa-solid fa-code mr-2"></i>Got: ${j.stdout || '(no output)'}</div>`;
            if (j.stderr) {
                consoleContent += `<div class="console-line console-error"><i class="fa-solid fa-bug mr-2"></i>Errors:\n${j.stderr}</div>`;
            }
            document.getElementById('question-card').classList.add('glow-error');
        }

        document.getElementById('console').innerHTML = consoleContent;
        setTimeout(() => {
            document.getElementById('question-card').classList.remove('glow-success', 'glow-error');
        }, 2000);

        if (j.ok && typeof j.score !== 'undefined') {
            score = j.score;
        }

    } catch (error) {
        document.getElementById('console').innerHTML = `<div class="console-line console-error"><i class="fa-solid fa-circle-exclamation mr-2"></i>Network error: ${error.message}</div>`;
    } finally {
        submitBtn.innerHTML = '<i class="fa-solid fa-paper-plane"></i> Submit Answer';
        submitBtn.disabled = false;
        isRunning = false;
    }
}

async function resetScore() {
    try {
        await fetch(location.pathname + '?action=reset', { method: 'POST' });
        score = 0;
        document.getElementById('console').innerHTML = `<div class="console-line"><i class="fa-solid fa-rotate mr-2"></i>Score telah diatur ulang menjadi 0</div>`;
    } catch (error) {
        document.getElementById('console').innerHTML = `<div class="console-line console-error"><i class="fa-solid fa-circle-exclamation mr-2"></i>Gagal mengatur ulang skor: ${error.message}</div>`;
    }
}

function next() {
    current = (current + 1) % QUESTIONS.length;
    const newPage = Math.floor(current / questionsPerPage);
    if (newPage !== currentPage) {
        currentPage = newPage;
    }
    renderQuestion(current);
}

function prev() {
    current = (current - 1 + QUESTIONS.length) % QUESTIONS.length;
    const newPage = Math.floor(current / questionsPerPage);
    if (newPage !== currentPage) {
        currentPage = newPage;
    }
    renderQuestion(current);
}

function nextPage() {
    const totalPages = Math.ceil(QUESTIONS.length / questionsPerPage);
    if (currentPage < totalPages - 1) { currentPage++; highlightList(); }
}

function prevPage() {
    if (currentPage > 0) { currentPage--; highlightList(); }
}

window.addEventListener('DOMContentLoaded', () => {
    initEditor();
    renderQuestion(0);

    document.getElementById('run-btn').addEventListener('click', runHandler);
    document.getElementById('submit-btn').addEventListener('click', submitHandler);
    document.getElementById('next-btn').addEventListener('click', next);
    document.getElementById('prev-btn').addEventListener('click', prev);
    document.getElementById('reset-score').addEventListener('click', resetScore);
    document.getElementById('page-prev-btn').addEventListener('click', prevPage);
    document.getElementById('page-next-btn').addEventListener('click', nextPage);

    // Fitur Anti-Mencontek: Mendeteksi jika pengguna beralih tab atau jendela.
    window.addEventListener('blur', () => {
        cheatCounter++;
        logWarning(`Peringatan #${cheatCounter}: Beralih dari halaman kuis terdeteksi. Harap tetap fokus.`);
    });

    document.getElementById('lang').addEventListener('change', () => {
        setModeForLang(document.getElementById('lang').value);
        renderQuestion(current);
    });
});